import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import * as Yup from 'yup';

export interface StudentFormData {
  name: string;
  title?: string;
  grade: string;
  year?: string;
  story: string;
  openingQuote?: string;
  closingQuote?: string;
  achievements?: string;
  images?: string[];
  featured?: boolean;
  category?: string;
  supportNeeded?: string;
}

export interface Student extends StudentFormData {
  id: string;
  createdAt: Date;
  updatedAt: Date;
}

export const studentSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(3).max(100),
  title: Yup.string().max(200, 'Title must not exceed 200 characters').nullable(),
  grade: Yup.string().required('Grade is required').min(2).max(50),
  year: Yup.string().max(4, 'Year must be 4 characters').nullable(),
  story: Yup.string().required('Story is required').min(50).max(50000),
  openingQuote: Yup.string().max(500, 'Opening quote must not exceed 500 characters').nullable(),
  closingQuote: Yup.string().max(500, 'Closing quote must not exceed 500 characters').nullable(),
  achievements: Yup.string().nullable().max(2000),
  images: Yup.array()
    .of(Yup.string().required('Image path is required').min(1, 'Image path cannot be empty'))
    .max(3, 'Maximum 3 images allowed')
    .nullable(),
  featured: Yup.boolean(),
  category: Yup.string().oneOf(['academic', 'sports', 'arts', 'other', ''], 'Invalid category').nullable(),
  supportNeeded: Yup.string().max(500).nullable(),
});

interface ApiResponse<T = Record<string, unknown>> {
  success: boolean;
  message: string;
  data?: T;
  errors?: Record<string, string>;
}

const fetchStudents = async (): Promise<Student[]> => {
  const response = await fetch('/api/admin/students');
  const data = await response.json();
  if (!response.ok) throw new Error(data.message || 'Failed to fetch students');
  return data.data;
};

const createStudentApi = async (values: StudentFormData): Promise<ApiResponse> => {
  const response = await fetch('/api/admin/students', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(values),
  });
  const data = await response.json();
  if (!response.ok) throw new Error(data.message || 'Failed to create student');
  return data;
};

const updateStudentApi = async ({ id, values }: { id: string; values: StudentFormData }): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/students/${id}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(values),
  });
  const data = await response.json();
  if (!response.ok) throw new Error(data.message || 'Failed to update student');
  return data;
};

const deleteStudentApi = async (id: string): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/students/${id}`, {
    method: 'DELETE',
  });
  const data = await response.json();
  if (!response.ok) throw new Error(data.message || 'Failed to delete student');
  return data;
};

export const useStudents = () => {
  return useQuery({
    queryKey: ['students'],
    queryFn: fetchStudents,
  });
};

export const useCreateStudent = () => {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: createStudentApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['students'] });
    },
  });
};

export const useUpdateStudent = () => {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: updateStudentApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['students'] });
    },
  });
};

export const useDeleteStudent = () => {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: deleteStudentApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['students'] });
    },
  });
};
