import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import * as Yup from 'yup';

export interface TeacherFormData {
  name: string;
  title: string;
  position: string;
  subject?: string;
  gender?: string;
  bio?: string;
  qualifications?: string;
  experience?: number;
  image?: string;
  email?: string;
  phone?: string;
  active?: boolean;
}

export interface Teacher extends TeacherFormData {
  id: string;
  slug: string;
  createdAt: Date;
  updatedAt: Date;
}

export const teacherSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(3).max(100),
  title: Yup.string().required('Title is required').min(2).max(50),
  position: Yup.string().required('Position/Subject is required').min(2).max(100),
  subject: Yup.string().max(100),
  gender: Yup.string().oneOf(['male', 'female', ''], 'Invalid gender'),
  bio: Yup.string().max(2000),
  qualifications: Yup.string().max(2000),
  experience: Yup.number().min(0).max(70),
  image: Yup.string().nullable().test('is-valid-image-url', 'Invalid image URL', (value) => {
    if (!value) return true;
    return value.startsWith('/') || value.startsWith('http://') || value.startsWith('https://');
  }),
  email: Yup.string().email('Invalid email').nullable(),
  phone: Yup.string().max(20).nullable(),
  active: Yup.boolean(),
});

interface ApiResponse<T = Record<string, unknown>> {
  success: boolean;
  message: string;
  data?: T;
  errors?: Record<string, string>;
}

const fetchTeachers = async (): Promise<Teacher[]> => {
  const response = await fetch('/api/admin/teachers');
  const data = await response.json();
  if (!response.ok) throw new Error(data.message || 'Failed to fetch teachers');
  return data.data;
};

const createTeacherApi = async (values: TeacherFormData): Promise<ApiResponse> => {
  const response = await fetch('/api/admin/teachers', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(values),
  });
  const data = await response.json();
  if (!response.ok) throw new Error(data.message || 'Failed to create teacher');
  return data;
};

const updateTeacherApi = async (id: string, values: TeacherFormData): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/teachers/${id}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(values),
  });
  const data = await response.json();
  if (!response.ok) throw new Error(data.message || 'Failed to update teacher');
  return data;
};

const deleteTeacherApi = async (id: string): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/teachers/${id}`, {
    method: 'DELETE',
  });
  const data = await response.json();
  if (!response.ok) throw new Error(data.message || 'Failed to delete teacher');
  return data;
};

export const useTeachers = () => {
  return useQuery({
    queryKey: ['teachers'],
    queryFn: fetchTeachers,
  });
};

export const useCreateTeacher = () => {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: createTeacherApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['teachers'] });
    },
  });
};

export const useUpdateTeacher = () => {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: ({ id, values }: { id: string; values: TeacherFormData }) => updateTeacherApi(id, values),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['teachers'] });
    },
  });
};

export const useDeleteTeacher = () => {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: deleteTeacherApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['teachers'] });
    },
  });
};
