/**
 * CSRF Token Generation and Verification
 * Uses HMAC-based tokens (no database required)
 */

import crypto from 'crypto';

/**
 * Generate a CSRF token
 * Token format: base64(random) + ':' + hmac
 * This is stateless - no server-side storage needed
 */
export function generateCsrfToken(secret: string = process.env.CSRF_SECRET || 'default-csrf-secret'): string {
  // Generate random bytes
  const random = crypto.randomBytes(32).toString('base64url');
  
  // Create HMAC signature
  const hmac = crypto
    .createHmac('sha256', secret)
    .update(random)
    .digest('base64url');
  
  // Return token as random:hmac
  return `${random}.${hmac}`;
}

/**
 * Verify a CSRF token
 * Returns true if token is valid, false otherwise
 */
export function verifyCsrfToken(token: string, secret: string = process.env.CSRF_SECRET || 'default-csrf-secret'): boolean {
  try {
    if (!token || typeof token !== 'string') {
      return false;
    }

    // Split token into parts
    const parts = token.split('.');
    if (parts.length !== 2) {
      return false;
    }

    const [random, providedHmac] = parts;

    // Regenerate HMAC with same random value
    const expectedHmac = crypto
      .createHmac('sha256', secret)
      .update(random)
      .digest('base64url');

    // Compare using timing-safe comparison
    return crypto.timingSafeEqual(
      Buffer.from(providedHmac),
      Buffer.from(expectedHmac)
    );
  } catch (error) {
    console.error('CSRF token verification error:', error);
    return false;
  }
}

/**
 * Extract CSRF token from request
 * Checks: header, body, or cookie
 */
export function getCsrfTokenFromRequest(request: Request): string | null {
  // Check X-CSRF-Token header (most secure)
  let token = request.headers.get('x-csrf-token');
  if (token) return token;

  // Check X-XSRF-Token header (alternative)
  token = request.headers.get('x-xsrf-token');
  if (token) return token;

  return null;
}

/**
 * Extract CSRF token from FormData body
 */
export function getCsrfTokenFromFormData(formData: FormData): string | null {
  return formData.get('_csrf') as string | null;
}

/**
 * Middleware to verify CSRF token on protected requests
 * Usage: const valid = verifyCsrfOnRequest(request);
 */
export async function verifyCsrfOnRequest(
  request: Request,
  method: string = request.method
): Promise<{ valid: boolean; error?: string }> {
  // Skip CSRF check for GET requests (safe methods)
  if (method === 'GET' || method === 'HEAD' || method === 'OPTIONS') {
    return { valid: true };
  }

  // For state-changing methods, verify CSRF token
  try {
    let token = getCsrfTokenFromRequest(request);

    // If not in headers, try to extract from body
    if (!token && (method === 'POST' || method === 'PUT' || method === 'DELETE')) {
      const contentType = request.headers.get('content-type') || '';
      
      if (contentType.includes('application/json')) {
        try {
          const body = await request.json();
          token = body._csrf || body.csrfToken;
        } catch {
          // Ignore JSON parsing errors
        }
      }
    }

    if (!token) {
      return {
        valid: false,
        error: 'CSRF token missing from request',
      };
    }

    const isValid = verifyCsrfToken(token);
    if (!isValid) {
      return {
        valid: false,
        error: 'Invalid or expired CSRF token',
      };
    }

    return { valid: true };
  } catch (error) {
    console.error('CSRF verification error:', error);
    return {
      valid: false,
      error: 'CSRF verification failed',
    };
  }
}

/**
 * Create CSRF error response
 */
export function createCsrfErrorResponse(message: string = 'CSRF token validation failed') {
  return new Response(
    JSON.stringify({
      success: false,
      message,
    }),
    {
      status: 403, // Forbidden
      headers: {
        'Content-Type': 'application/json',
      },
    }
  );
}
