/**
 * Page-Level Permission Configuration
 * Defines all admin pages with granular permission control
 */

export interface PagePermission {
  id: string;
  name: string;
  description: string;
  path: string;
  category: 'content' | 'management' | 'system';
  actions: {
    id: string;
    label: string;
    description: string;
  }[];
  adminOnly?: boolean; // Only super admins can grant access
}

export const ADMIN_PAGES_CONFIG: PagePermission[] = [
  // CONTENT PAGES
  {
    id: 'news',
    name: 'News & Articles',
    description: 'Manage news articles and blog posts',
    path: '/admin/news',
    category: 'content',
    actions: [
      { id: 'view', label: 'View', description: 'Can view news list and details' },
      { id: 'create', label: 'Create', description: 'Can create new articles' },
      { id: 'edit', label: 'Edit', description: 'Can edit existing articles' },
      { id: 'delete', label: 'Delete', description: 'Can delete articles' },
      { id: 'publish', label: 'Publish', description: 'Can publish/unpublish articles' },
    ],
  },
  {
    id: 'teachers',
    name: 'Teachers',
    description: 'Manage teacher profiles and information',
    path: '/admin/teachers',
    category: 'content',
    actions: [
      { id: 'view', label: 'View', description: 'Can view teacher list' },
      { id: 'create', label: 'Create', description: 'Can add new teachers' },
      { id: 'edit', label: 'Edit', description: 'Can edit teacher profiles' },
      { id: 'delete', label: 'Delete', description: 'Can remove teachers' },
    ],
  },
  {
    id: 'leadership',
    name: 'Leadership Team',
    description: 'Manage leadership and staff profiles',
    path: '/admin/leadership',
    category: 'content',
    actions: [
      { id: 'view', label: 'View', description: 'Can view leadership list' },
      { id: 'create', label: 'Create', description: 'Can add leadership members' },
      { id: 'edit', label: 'Edit', description: 'Can edit leadership profiles' },
      { id: 'delete', label: 'Delete', description: 'Can remove leadership members' },
    ],
  },
  {
    id: 'students',
    name: 'Student Stories',
    description: 'Manage student achievement stories',
    path: '/admin/students',
    category: 'content',
    actions: [
      { id: 'view', label: 'View', description: 'Can view student stories' },
      { id: 'create', label: 'Create', description: 'Can create new stories' },
      { id: 'edit', label: 'Edit', description: 'Can edit existing stories' },
      { id: 'delete', label: 'Delete', description: 'Can delete stories' },
      { id: 'feature', label: 'Feature', description: 'Can mark stories as featured' },
    ],
  },

  // MANAGEMENT PAGES
  {
    id: 'contacts',
    name: 'Contact Messages',
    description: 'View and respond to contact form submissions',
    path: '/admin/contacts',
    category: 'management',
    actions: [
      { id: 'view', label: 'View', description: 'Can view contact messages' },
      { id: 'reply', label: 'Reply', description: 'Can reply to messages' },
      { id: 'delete', label: 'Delete', description: 'Can delete messages' },
      { id: 'export', label: 'Export', description: 'Can export contact data' },
    ],
  },
  {
    id: 'admissions',
    name: 'Admissions',
    description: 'Manage admission inquiries and applications',
    path: '/admin/admissions',
    category: 'management',
    actions: [
      { id: 'view', label: 'View', description: 'Can view admission inquiries' },
      { id: 'edit', label: 'Edit', description: 'Can update inquiry status' },
      { id: 'delete', label: 'Delete', description: 'Can delete inquiries' },
      { id: 'export', label: 'Export', description: 'Can export admission data' },
    ],
  },
  {
    id: 'donations',
    name: 'Donations',
    description: 'Track and manage donations',
    path: '/admin/donations',
    category: 'management',
    actions: [
      { id: 'view', label: 'View', description: 'Can view donation records' },
      { id: 'create', label: 'Create', description: 'Can add manual donations' },
      { id: 'edit', label: 'Edit', description: 'Can update donation info' },
      { id: 'delete', label: 'Delete', description: 'Can delete donation records' },
      { id: 'export', label: 'Export', description: 'Can export donation data' },
    ],
  },
  {
    id: 'newsletter',
    name: 'Newsletter',
    description: 'Manage newsletter subscribers',
    path: '/admin/newsletter',
    category: 'management',
    actions: [
      { id: 'view', label: 'View', description: 'Can view subscriber list' },
      { id: 'export', label: 'Export', description: 'Can export subscribers' },
      { id: 'delete', label: 'Delete', description: 'Can remove subscribers' },
    ],
  },

  // SYSTEM PAGES
  {
    id: 'users',
    name: 'User Management',
    description: 'Manage admin users and their permissions',
    path: '/admin/users',
    category: 'system',
    adminOnly: true,
    actions: [
      { id: 'view', label: 'View', description: 'Can view user list' },
      { id: 'approve', label: 'Approve', description: 'Can approve pending users' },
      { id: 'reject', label: 'Reject', description: 'Can reject users' },
      { id: 'suspend', label: 'Suspend', description: 'Can suspend users' },
      { id: 'edit_permissions', label: 'Edit Permissions', description: 'Can modify user permissions' },
      { id: 'delete', label: 'Delete', description: 'Can delete users' },
    ],
  },
  {
    id: 'settings',
    name: 'Settings',
    description: 'System configuration and settings',
    path: '/admin/settings',
    category: 'system',
    adminOnly: true,
    actions: [
      { id: 'view', label: 'View', description: 'Can view settings' },
      { id: 'edit', label: 'Edit', description: 'Can modify settings' },
    ],
  },
  {
    id: 'dashboard',
    name: 'Dashboard',
    description: 'Overview and analytics dashboard',
    path: '/admin',
    category: 'system',
    actions: [
      { id: 'view', label: 'View', description: 'Can access dashboard' },
    ],
  },
];

// Type for user's custom permissions stored in database
export interface UserPagePermissions {
  [pageId: string]: string[]; // pageId -> array of action ids
}

// Helper to check if user has access to a page
export function hasPageAccess(
  customPermissions: UserPagePermissions | null,
  role: string,
  pageId: string,
  action?: string
): boolean {
  // Super admin always has access
  if (role === 'admin') return true;

  // Dashboard is accessible to everyone (shows different content based on permissions)
  if (pageId === 'dashboard') return true;

  // No custom permissions - deny access (must be explicitly granted)
  if (!customPermissions || !customPermissions[pageId]) return false;

  const pagePermissions = customPermissions[pageId];

  // Check if page is accessible
  if (pagePermissions.length === 0) return false;

  // If no specific action required, just check page access
  if (!action) return true;

  // Check specific action permission
  return pagePermissions.includes(action);
}

// Helper to get all pages user can access
export function getUserAccessiblePages(
  customPermissions: UserPagePermissions | null,
  role: string
): string[] {
  // Super admin can access everything
  if (role === 'admin') {
    return ADMIN_PAGES_CONFIG.map((page) => page.id);
  }

  // No custom permissions - no access
  if (!customPermissions) return [];

  // Return pages with at least one permission
  return Object.keys(customPermissions).filter(
    (pageId) => customPermissions[pageId]?.length > 0
  );
}

// Helper to get permission summary for a user
export function getPermissionSummary(
  customPermissions: UserPagePermissions | null,
  role: string
): {
  pageId: string;
  pageName: string;
  actions: string[];
}[] {
  if (role === 'admin') {
    return ADMIN_PAGES_CONFIG.map((page) => ({
      pageId: page.id,
      pageName: page.name,
      actions: ['All'],
    }));
  }

  if (!customPermissions) return [];

  return ADMIN_PAGES_CONFIG.filter((page) => customPermissions[page.id]?.length > 0).map(
    (page) => ({
      pageId: page.id,
      pageName: page.name,
      actions: customPermissions[page.id] || [],
    })
  );
}

// Group pages by category
export function getPagesByCategory() {
  return {
    content: ADMIN_PAGES_CONFIG.filter((p) => p.category === 'content'),
    management: ADMIN_PAGES_CONFIG.filter((p) => p.category === 'management'),
    system: ADMIN_PAGES_CONFIG.filter((p) => p.category === 'system'),
  };
}
