/**
 * MVP Permission System
 * Simple role-based permissions with optional custom overrides
 */

export type PermissionAction = 'view' | 'manage';
export type PermissionModule =
  | 'teachers'
  | 'news'
  | 'students'
  | 'admissions'
  | 'donations'
  | 'contacts'
  | 'newsletter'
  | 'settings'
  | 'users';

export type Permission = {
  module: PermissionModule | '*';
  action: PermissionAction;
};

export type UserRole = 'admin' | 'editor' | 'viewer';
export const ROLES: UserRole[] = ['admin', 'editor', 'viewer'];
/**
 * Default permissions for each role
 */
export const ROLE_PERMISSIONS: Record<UserRole, Permission[]> = {
  // Admin: Full access to everything
  admin: [{ module: '*', action: 'manage' }],

  // Editor: Can manage content, view administrative data
  editor: [
    { module: 'teachers', action: 'manage' },
    { module: 'news', action: 'manage' },
    { module: 'students', action: 'manage' },
    { module: 'admissions', action: 'view' },
    { module: 'donations', action: 'view' },
    { module: 'contacts', action: 'view' },
    { module: 'newsletter', action: 'view' },
    { module: 'settings', action: 'view' },
    { module: 'users', action: 'view' },
  ],

  // Viewer: Read-only access to everything
  viewer: [{ module: '*', action: 'view' }],
};

/**
 * Check if a user has permission to perform an action on a module
 * 
 * @param userRole - The user's role (admin/editor/viewer)
 * @param module - The module to check (teachers/news/etc)
 * @param action - The action to perform (view/manage)
 * @param customPermissions - Optional custom permission overrides (JSON field from DB)
 * @returns boolean - true if user has permission, false otherwise
 */
export function hasPermission(
  userRole: string,
  module: PermissionModule,
  action: PermissionAction,
  customPermissions?: Record<string, PermissionAction[]> | null
): boolean {
  // Check custom permissions first (if provided)
  if (customPermissions && customPermissions[module]) {
    const allowedActions = customPermissions[module];
    // If user has 'manage', they can do both 'manage' and 'view'
    if (allowedActions.includes('manage')) return true;
    if (action === 'view' && allowedActions.includes('view')) return true;
    // If custom permissions exist for this module but don't include the action, deny
    return false;
  }

  // Fall back to role-based permissions
  const permissions = ROLE_PERMISSIONS[userRole as UserRole] || [];

  // Check for wildcard admin access
  if (permissions.some((p) => p.module === '*' && p.action === 'manage')) {
    return true;
  }

  // Check for wildcard view access (viewer role)
  if (
    action === 'view' &&
    permissions.some((p) => p.module === '*' && p.action === 'view')
  ) {
    return true;
  }

  // Check for specific module permission
  return permissions.some((p) => {
    const moduleMatch = p.module === module || p.module === '*';
    const actionMatch =
      p.action === action || (p.action === 'manage' && action === 'view');
    return moduleMatch && actionMatch;
  });
}

/**
 * Get all permissions for a user
 * 
 * @param userRole - The user's role
 * @param customPermissions - Optional custom permission overrides
 * @returns Array of permissions
 */
export function getUserPermissions(
  userRole: string,
  customPermissions?: Record<string, PermissionAction[]> | null
): Permission[] {
  const basePermissions = ROLE_PERMISSIONS[userRole as UserRole] || [];
  
  if (!customPermissions) {
    return basePermissions;
  }

  // Merge custom permissions with base permissions
  const merged: Permission[] = [...basePermissions];
  
  Object.entries(customPermissions).forEach(([module, actions]) => {
    actions.forEach((action) => {
      // Only add if not already covered by base permissions
      const alreadyHas = basePermissions.some(
        (p) =>
          (p.module === module || p.module === '*') &&
          (p.action === action || p.action === 'manage')
      );
      
      if (!alreadyHas) {
        merged.push({ module: module as PermissionModule, action });
      }
    });
  });

  return merged;
}

type CustomPermissions = Record<string, PermissionAction[]> | null | undefined;

/**
 * Check if user can perform a specific action
 * Commonly used actions mapped to permission checks
 */
export const can = {
  // Teachers
  viewTeachers: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'teachers', 'view', custom),
  manageTeachers: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'teachers', 'manage', custom),

  // News
  viewNews: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'news', 'view', custom),
  manageNews: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'news', 'manage', custom),

  // Students
  viewStudents: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'students', 'view', custom),
  manageStudents: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'students', 'manage', custom),

  // Admissions
  viewAdmissions: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'admissions', 'view', custom),
  manageAdmissions: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'admissions', 'manage', custom),

  // Donations
  viewDonations: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'donations', 'view', custom),
  manageDonations: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'donations', 'manage', custom),

  // Contacts
  viewContacts: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'contacts', 'view', custom),
  manageContacts: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'contacts', 'manage', custom),

  // Newsletter
  viewNewsletter: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'newsletter', 'view', custom),
  manageNewsletter: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'newsletter', 'manage', custom),

  // Settings
  viewSettings: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'settings', 'view', custom),
  manageSettings: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'settings', 'manage', custom),

  // Users
  viewUsers: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'users', 'view', custom),
  manageUsers: (role: string, custom?: CustomPermissions) =>
    hasPermission(role, 'users', 'manage', custom),
};

/**
 * Role display helpers
 */
export const roleLabels: Record<UserRole, string> = {
  admin: 'Administrator',
  editor: 'Content Editor',
  viewer: 'Viewer',
};

export const roleDescriptions: Record<UserRole, string> = {
  admin: 'Full access to all features and settings',
  editor: 'Can manage content (teachers, news, students)',
  viewer: 'Read-only access to all content',
};

export const roleBadgeColors: Record<UserRole, string> = {
  admin: 'bg-red-100 text-red-800',
  editor: 'bg-blue-100 text-blue-800',
  viewer: 'bg-gray-100 text-gray-800',
};
