/**
 * Rate Limiter for Admin API Protection
 * In-memory implementation for single-instance deployments
 * For distributed systems, use Redis
 */

interface RateLimitRecord {
  count: number;
  resetTime: number;
}

// Store: key => { count, resetTime }
const rateLimitStore = new Map<string, RateLimitRecord>();

// Cleanup interval - remove expired records every 5 minutes
setInterval(() => {
  const now = Date.now();
  for (const [key, record] of rateLimitStore.entries()) {
    if (record.resetTime < now) {
      rateLimitStore.delete(key);
    }
  }
}, 5 * 60 * 1000);

/**
 * Check if request exceeds rate limit
 * @param key - Unique identifier (e.g., IP address, user ID, email)
 * @param limit - Max requests allowed
 * @param windowMs - Time window in milliseconds (default: 60000 = 1 minute)
 * @returns { allowed: boolean, remaining: number, resetTime: number }
 */
export function checkRateLimit(
  key: string,
  limit: number = 100,
  windowMs: number = 60 * 1000
): { allowed: boolean; remaining: number; resetTime: number } {
  const now = Date.now();
  const record = rateLimitStore.get(key);

  // First request or window expired
  if (!record || record.resetTime < now) {
    const newRecord: RateLimitRecord = {
      count: 1,
      resetTime: now + windowMs,
    };
    rateLimitStore.set(key, newRecord);
    return {
      allowed: true,
      remaining: limit - 1,
      resetTime: newRecord.resetTime,
    };
  }

  // Increment counter
  record.count++;

  const remaining = Math.max(0, limit - record.count);
  const allowed = record.count <= limit;

  return {
    allowed,
    remaining,
    resetTime: record.resetTime,
  };
}

/**
 * Rate limit presets for different scenarios
 */
export const RateLimitPresets = {
  /**
   * Strict limits for authentication endpoints
   * 5 attempts per minute per IP
   */
  LOGIN: {
    limit: 5,
    windowMs: 60 * 1000, // 1 minute
    message: 'Too many login attempts. Please try again later.',
  },

  /**
   * Moderate limits for password change
   * 3 attempts per 5 minutes per user
   */
  PASSWORD_CHANGE: {
    limit: 3,
    windowMs: 5 * 60 * 1000, // 5 minutes
    message: 'Too many password change attempts. Please try again later.',
  },

  /**
   * Standard limits for admin API routes
   * 100 requests per minute per user
   */
  ADMIN_API: {
    limit: 100,
    windowMs: 60 * 1000, // 1 minute
    message: 'Too many requests. Please try again later.',
  },

  /**
   * Relaxed limits for read-only operations
   * 500 requests per minute per user
   */
  READ_ONLY: {
    limit: 500,
    windowMs: 60 * 1000, // 1 minute
    message: 'Too many requests. Please try again later.',
  },

  /**
   * Very strict for file uploads
   * 10 uploads per 5 minutes per user
   */
  FILE_UPLOAD: {
    limit: 10,
    windowMs: 5 * 60 * 1000, // 5 minutes
    message: 'Too many upload attempts. Please try again later.',
  },
};

/**
 * Create rate limit error response
 */
export function createRateLimitResponse(
  preset: (typeof RateLimitPresets)[keyof typeof RateLimitPresets],
  retryAfter: number
) {
  return new Response(
    JSON.stringify({
      success: false,
      message: preset.message,
      retryAfter,
    }),
    {
      status: 429, // Too Many Requests
      headers: {
        'Retry-After': Math.ceil(retryAfter / 1000).toString(),
        'Content-Type': 'application/json',
      },
    }
  );
}

/**
 * Extract IP address from request
 */
export function getClientIp(request: Request): string {
  const forwarded = request.headers.get('x-forwarded-for');
  if (forwarded) {
    return forwarded.split(',')[0].trim();
  }
  return request.headers.get('x-real-ip') || 'unknown';
}

/**
 * Get rate limit key for login attempts (IP-based)
 */
export function getLoginRateLimitKey(ip: string): string {
  return `login:${ip}`;
}

/**
 * Get rate limit key for user API calls (user ID-based)
 */
export function getApiRateLimitKey(userId: string, endpoint?: string): string {
  return `api:${userId}${endpoint ? `:${endpoint}` : ''}`;
}

/**
 * Get rate limit key for password changes (user ID-based)
 */
export function getPasswordRateLimitKey(userId: string): string {
  return `password:${userId}`;
}

/**
 * Get rate limit key for file uploads (user ID-based)
 */
export function getUploadRateLimitKey(userId: string): string {
  return `upload:${userId}`;
}
