/**
 * Admin API Security Utilities
 * Handles authentication, authorization, validation, and rate limiting for admin API routes
 */

import { NextRequest, NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import {
  hasPermission,
  PermissionModule,
  PermissionAction,
} from './permissions';
import {
  RateLimitPresets,
  createRateLimitResponse,
  getApiRateLimitKey,
} from './rate-limiter';

export interface AdminApiResponse<T = Record<string, unknown>> {
  success: boolean;
  message: string;
  data?: T;
  errors?: Record<string, string>;
}

/**
 * Verify admin authentication for API routes
 * Returns user session or null if not authenticated
 * Use in API route handlers as first check
 */
export async function verifyAdminAuth() {
  try {
    // Get session from auth
    const session = await auth();

    if (!session?.user?.id) {
      return {
        authenticated: false,
        authorized: false,
        error: 'Unauthorized - Please sign in',
        status: 401,
      };
    }

    // Check if user has admin or editor role
    const userRole = session.user.role as string;
    if (userRole !== 'admin' && userRole !== 'editor') {
      return {
        authenticated: true,
        authorized: false,
        error: 'Forbidden - Admin access required',
        status: 403,
      };
    }

    return {
      authenticated: true,
      authorized: true,
      user: session.user,
      userId: session.user.id,
      userRole: session.user.role,
      error: null,
    };
  } catch (error) {
    console.error('Auth verification error:', error);
    return {
      authenticated: false,
      authorized: false,
      error: 'Authentication error',
      status: 401,
    };
  }
}

/**
 * Check admin-only permission (not just editor)
 */
export function requireAdminOnly(userRole?: string): boolean {
  return userRole === 'admin';
}

/**
 * Check admin/editor permission
 */
export function requireAdminAccess(userRole?: string): boolean {
  return userRole === 'admin' || userRole === 'editor';
}

/**
 * Check if user has specific permission
 * @param userRole - User's role (admin/editor/viewer)
 * @param module - Module to check (teachers/news/etc)
 * @param action - Action to perform (view/manage)
 * @param customPermissions - Optional custom permissions from DB
 * @returns boolean - true if user has permission
 */
export async function checkPermission(
  userRole: string,
  module: PermissionModule,
  action: PermissionAction,
  customPermissions?: Record<string, PermissionAction[]> | null
): Promise<boolean> {
  return hasPermission(userRole, module, action, customPermissions);
}

/**
 * Get user with permissions from database
 * Used for permission-based authorization
 */
export async function getUserWithPermissions(userId: string) {
  const user = await prisma.adminUser.findUnique({
    where: { id: userId },
    select: {
      id: true,
      email: true,
      name: true,
      role: true,
      status: true,
      active: true,
      permissions: true,
    },
  });

  if (!user || !user.active || user.status !== 'approved') {
    return null;
  }

  return user;
}

/**
 * Create standardized error response for API routes
 */
export function createErrorResponse(
  message: string,
  status: number = 400,
  errors?: Record<string, string>
): NextResponse<AdminApiResponse> {
  return NextResponse.json(
    {
      success: false,
      message,
      ...(errors && { errors }),
    },
    { status }
  );
}

/**
 * Create standardized success response for API routes
 */
export function createSuccessResponse<T>(
  data: T,
  message: string = 'Success',
  status: number = 200
): NextResponse<AdminApiResponse<T>> {
  return NextResponse.json(
    {
      success: true,
      message,
      data,
    },
    { status }
  );
}

/**
 * Middleware for API routes - handles auth and error responses
 * Usage: const auth = await withAdminAuth();
 *        if (auth.error) return auth.response!;
 */
export async function withAdminAuth() {
  const authResult = await verifyAdminAuth();

  if (!authResult.authorized) {
    return {
      error: true,
      response: createErrorResponse(
        authResult.error || 'Unauthorized',
        authResult.status || 401
      ),
      user: null,
    };
  }

  return {
    error: false,
    response: null,
    user: authResult.user,
    userId: authResult.userId,
    userRole: authResult.userRole,
  };
}

/**
 * Middleware for API routes with permission check
 * Usage: const auth = await withPermission('news', 'manage');
 *        if (auth.error) return auth.response!;
 */
export async function withPermission(
  module: PermissionModule,
  action: PermissionAction
) {
  const authResult = await verifyAdminAuth();

  if (!authResult.authorized) {
    return {
      error: true,
      response: createErrorResponse(
        authResult.error || 'Unauthorized',
        authResult.status || 401
      ),
      user: null,
    };
  }

  // Get user with permissions from database
  const user = await getUserWithPermissions(authResult.userId as string);
  
  if (!user) {
    return {
      error: true,
      response: createErrorResponse('User not found or inactive', 403),
      user: null,
    };
  }

  // Check permission
  const permitted = hasPermission(
    user.role,
    module,
    action,
    user.permissions as Record<string, PermissionAction[]> | null
  );

  if (!permitted) {
    return {
      error: true,
      response: createErrorResponse(
        `Forbidden - You don't have permission to ${action} ${module}`,
        403
      ),
      user: null,
    };
  }

  return {
    error: false,
    response: null,
    user: authResult.user,
    userId: authResult.userId,
    userRole: authResult.userRole,
  };
}

/**
 * Rate-limited API auth middleware
 * Includes both authentication and rate limiting checks
 * Usage: const auth = await withAdminAuthAndRateLimit(request, limit, windowMs);
 */
export async function withAdminAuthAndRateLimit(
  request: NextRequest,
  limit: number = 100,
  windowMs: number = 60 * 1000
) {
  const authResult = await verifyAdminAuth();

  if (!authResult.authorized) {
    return {
      error: true,
      response: createErrorResponse(
        authResult.error || 'Unauthorized',
        authResult.status || 401
      ),
      user: null,
    };
  }

  // Check rate limit for authenticated user
  const rateLimitKey = getApiRateLimitKey(authResult.userId as string);
  const { checkRateLimit } = await import('./rate-limiter');
  const rateLimit = checkRateLimit(rateLimitKey, limit, windowMs);

  if (!rateLimit.allowed) {
    return {
      error: true,
      response: createRateLimitResponse(RateLimitPresets.ADMIN_API, rateLimit.resetTime - Date.now()),
      user: null,
    };
  }

  return {
    error: false,
    response: null,
    user: authResult.user,
    userId: authResult.userId,
    userRole: authResult.userRole,
    rateLimit: {
      remaining: rateLimit.remaining,
      resetTime: rateLimit.resetTime,
    },
  };
}

/**
 * Validate Content-Type header for API requests
 */
export function validateContentType(
  request: NextRequest,
  expectedType: string = 'application/json'
): boolean {
  const contentType = request.headers.get('content-type');
  return contentType?.includes(expectedType) ?? false;
}

/**
 * Rate limit check (basic implementation)
 * In production, use a proper rate limiting library
 */
const requestCounts = new Map<string, { count: number; resetTime: number }>();

export function checkRateLimit(
  identifier: string,
  limit: number = 60,
  windowMs: number = 60000 // 1 minute
): { allowed: boolean; remaining: number; resetTime: number } {
  const now = Date.now();
  const record = requestCounts.get(identifier);

  if (!record || now > record.resetTime) {
    // Reset or first request
    requestCounts.set(identifier, { count: 1, resetTime: now + windowMs });
    return { allowed: true, remaining: limit - 1, resetTime: now + windowMs };
  }

  if (record.count >= limit) {
    return { allowed: false, remaining: 0, resetTime: record.resetTime };
  }

  record.count++;
  return {
    allowed: true,
    remaining: limit - record.count,
    resetTime: record.resetTime,
  };
}

/**
 * Generate CSRF token (basic implementation)
 * In production, use a proper CSRF token library
 */
const csrfTokens = new Set<string>();

export function generateCsrfToken(): string {
  const token = Math.random().toString(36).substring(2, 15) +
                Math.random().toString(36).substring(2, 15);
  csrfTokens.add(token);
  return token;
}

/**
 * Verify CSRF token
 */
export function verifyCsrfToken(token: string): boolean {
  const isValid = csrfTokens.has(token);
  if (isValid) {
    csrfTokens.delete(token); // One-time use tokens
  }
  return isValid;
}

/**
 * Log admin action for audit trail
 */
export async function logAdminAction(
  userId: string,
  action: string,
  resource: string,
  resourceId: string,
  details?: Record<string, unknown>
) {
  // TODO: Implement proper audit logging to database
  console.log({
    timestamp: new Date().toISOString(),
    userId,
    action,
    resource,
    resourceId,
    details,
  });
}

/**
 * Validate request body against schema
 */
export async function validateRequestBody(
  request: NextRequest
) {
  try {
    const validated = await request.json();
    return { valid: true, data: validated, error: null };
  } catch (error: unknown) {
    const errorMessage = error instanceof Error ? error.message : 'Invalid request body';
    return {
      valid: false,
      data: null,
      error: errorMessage,
    };
  }
}
