import { PrismaAdapter } from "@auth/prisma-adapter";
import NextAuth, { CredentialsSignin } from "next-auth";
import CredentialsProvider from "next-auth/providers/credentials";
import { prisma } from "@/lib/prisma";
import bcrypt from "bcryptjs";

// Custom error classes for different account states
class AccountPendingError extends CredentialsSignin {
  code = "ACCOUNT_PENDING_APPROVAL";
}

class AccountRejectedError extends CredentialsSignin {
  code = "ACCOUNT_REJECTED";
}

class AccountSuspendedError extends CredentialsSignin {
  code = "ACCOUNT_SUSPENDED";
}

class AccountInactiveError extends CredentialsSignin {
  code = "ACCOUNT_INACTIVE";
}

declare module "next-auth" {
  interface Session {
    user: {
      id: string;
      email: string;
      name: string;
      role: string;
      avatar?: string;
    };
  }
}

export const { handlers, auth, signIn, signOut } = NextAuth({
  adapter: PrismaAdapter(prisma),
  providers: [
    CredentialsProvider({
      name: "Credentials",
      credentials: {
        email: {
          label: "Email",
          type: "email",
          placeholder: "admin@school.com",
        },
        password: { label: "Password", type: "password" },
      },
      async authorize(credentials) {
        if (!credentials?.email || !credentials?.password) {
          return null;
        }

        const user = await prisma.adminUser.findUnique({
          where: { email: credentials.email as string },
        });

        if (!user) {
          return null;
        }

        const passwordMatch = await bcrypt.compare(
          credentials.password as string,
          user.password
        );

        if (!passwordMatch) {
          return null;
        }

        // Check user approval status - throw custom errors
        if (user.status === "pending") {
          throw new AccountPendingError();
        }

        if (user.status === "rejected") {
          throw new AccountRejectedError();
        }

        if (user.status === "suspended") {
          throw new AccountSuspendedError();
        }

        if (!user.active) {
          throw new AccountInactiveError();
        }

        return {
          id: user.id,
          email: user.email,
          name: user.name,
          role: user.role,
          avatar: user.avatar,
        };
      },
    }),
  ],
  pages: {
    signIn: "/auth/signin",
    error: "/auth/error",
  },
  callbacks: {
    async jwt({ token, user }) {
      if (user) {
        token.id = user.id;
        token.role = (user as { role?: string; avatar?: string }).role;
        token.avatar = (user as { role?: string; avatar?: string }).avatar;
      }
      return token;
    },
    async session({ session, token }) {
      if (session.user) {
        session.user.id = token.id as string;
        session.user.role = token.role as string;
        session.user.avatar = token.avatar as string;
      }
      return session;
    },
  },
  session: {
    strategy: "jwt",
    maxAge: 30 * 24 * 60 * 60, // 30 days
  },
  events: {
    async signIn({ user }) {
      if (user.id) {
        await prisma.adminUser.update({
          where: { id: user.id },
          data: { lastLogin: new Date() },
        });
      }
    },
  },
});
