import nodemailer from "nodemailer";

// Verify SMTP configuration exists
const validateSMTPConfig = () => {
  const required = ['SMTP_HOST', 'SMTP_PORT', 'SMTP_USER', 'SMTP_PASS', 'SMTP_FROM_EMAIL'];
  const missing = required.filter(key => !process.env[key]);
  
  if (missing.length > 0) {
    console.warn(`⚠️ Missing SMTP configuration: ${missing.join(', ')}`);
    return false;
  }
  return true;
};

// Create reusable transporter
let transporter: nodemailer.Transporter | null = null;

const getTransporter = () => {
  if (transporter) return transporter;

  if (!validateSMTPConfig()) {
    throw new Error('SMTP configuration is incomplete. Check your .env.local file.');
  }

  transporter = nodemailer.createTransport({
    host: process.env.SMTP_HOST,
    port: parseInt(process.env.SMTP_PORT || '465'),
    secure: process.env.SMTP_PORT === '465', // true for 465, false for other ports
    auth: {
      user: process.env.SMTP_USER,
      pass: process.env.SMTP_PASS,
    },
  });

  return transporter;
};

interface EmailOptions {
  to: string;
  subject: string;
  html: string;
  text?: string;
  replyTo?: string;
  cc?: string[];
  bcc?: string[];
}

/**
 * Send email using shared hosting SMTP
 */
export async function sendEmail({ to, subject, html, text, replyTo, cc, bcc }: EmailOptions) {
  try {
    const transporter = getTransporter();
    
    const result = await transporter.sendMail({
      from: `"${process.env.SMTP_FROM_NAME || "Setaragan School"}" <${process.env.SMTP_FROM_EMAIL}>`,
      to,
      subject,
      html,
      text: text || html.replace(/<[^>]*>/g, ""), // Strip HTML for plain text fallback
      replyTo: replyTo || process.env.SMTP_FROM_EMAIL,
      cc,
      bcc,
    });

    console.log(`✅ Email sent to ${to}`);
    return { success: true, messageId: result.messageId };
  } catch (error) {
    console.error("Email sending error:", error);
    return { success: false, error: String(error) };
  }
}

/**
 * Send contact form confirmation email
 */
export async function sendContactConfirmation(
  email: string,
  name: string,
  subject: string
) {
  const html = `
    <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
      <h2 style="color: #1e40af;">Thank You for Contacting Us</h2>
      <p>Dear ${name},</p>
      <p>We have received your message with subject: <strong>${subject}</strong></p>
      <p>Our team will review your inquiry and get back to you as soon as possible.</p>
      <p>We typically respond within 24-48 hours during business days.</p>
      <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 20px 0;" />
      <p style="color: #6b7280; font-size: 12px;">
        This is an automated message. Please do not reply to this email.
      </p>
      <p style="color: #6b7280; font-size: 12px;">
        <strong>Setaragan Zirak School</strong><br>
        Market Safid Street, Kabul, Afghanistan<br>
        Phone: +93747500432
      </p>
    </div>
  `;

  return sendEmail({
    to: email,
    subject: "We received your message - Setaragan Zirak School",
    html,
  });
}

/**
 * Send contact form notification to admin
 */
export async function sendContactNotification(
  contactData: {
    name: string;
    email: string;
    phone?: string;
    subject: string;
    message: string;
  }
) {
  const html = `
    <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
      <h2 style="color: #1e40af;">New Contact Form Submission</h2>
      <div style="background: #f3f4f6; padding: 15px; border-radius: 5px; margin: 15px 0;">
        <p><strong>Name:</strong> ${contactData.name}</p>
        <p><strong>Email:</strong> <a href="mailto:${contactData.email}">${contactData.email}</a></p>
        ${contactData.phone ? `<p><strong>Phone:</strong> ${contactData.phone}</p>` : ''}
        <p><strong>Subject:</strong> ${contactData.subject}</p>
      </div>
      <h3 style="color: #374151;">Message:</h3>
      <div style="white-space: pre-wrap; background: #f9fafb; padding: 15px; border-left: 3px solid #3b82f6; border-radius: 3px;">
${contactData.message}
      </div>
      <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 20px 0;" />
      <p style="color: #6b7280; font-size: 12px;">
        Reply to: <a href="mailto:${contactData.email}">${contactData.email}</a>
      </p>
    </div>
  `;

  const recipientEmail = process.env.CONTACT_FORM_RECIPIENT || process.env.SMTP_FROM_EMAIL;

  return sendEmail({
    to: recipientEmail!,
    subject: `New Contact Form: ${contactData.subject}`,
    html,
    replyTo: contactData.email,
  });
}

/**
 * Send newsletter welcome email
 */
export async function sendNewsletterWelcome(
  email: string,
  firstName?: string
) {
  const html = `
    <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
      <h2 style="color: #1e40af;">Welcome to Our Newsletter!</h2>
      <p>Dear ${firstName || 'Subscriber'},</p>
      <p>Thank you for subscribing to our newsletter. You'll now receive updates about:</p>
      <ul style="color: #374151; line-height: 1.8;">
        <li>📰 Latest news and announcements</li>
        <li>🎓 Student achievements and events</li>
        <li>📅 Important school dates and updates</li>
        <li>🏆 Success stories and highlights</li>
      </ul>
      <p>We promise not to spam your inbox – only meaningful updates!</p>
      <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 20px 0;" />
      <p style="color: #6b7280; font-size: 12px;">
        To unsubscribe, click <a href="${process.env.NEXT_PUBLIC_APP_URL}/unsubscribe?email=${email}" style="color: #3b82f6;">here</a>
      </p>
    </div>
  `;

  return sendEmail({
    to: email,
    subject: "Welcome to Setaragan Zirak School Newsletter",
    html,
  });
}

/**
 * Send welcome email to new admin user
 */
export async function sendWelcomeEmail(email: string, name: string, password: string) {
  const html = `
    <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
      <h2 style="color: #1e40af;">Welcome to Admin Panel</h2>
      <p>Dear ${name},</p>
      <p>Your admin account has been created successfully!</p>
      <div style="background-color: #f3f4f6; padding: 15px; border-radius: 5px; margin: 20px 0;">
        <p><strong>Login Information:</strong></p>
        <p>Email: <code>${email}</code></p>
        <p>Temporary Password: <code>${password}</code></p>
        <p style="color: #dc2626; font-size: 12px;">Please change your password after first login.</p>
      </div>
      <p>
        <a href="${process.env.NEXTAUTH_URL}/auth/signin" style="background-color: #2563eb; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block;">
          Login to Admin Panel
        </a>
      </p>
      <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 20px 0;" />
      <p style="color: #6b7280; font-size: 12px;">
        If you didn't request this account, please contact the administrator.
      </p>
    </div>
  `;

  return sendEmail({
    to: email,
    subject: "Your Admin Account Created - Setaragan Zirak School",
    html,
  });
}

/**
 * Send donation thank you email
 */
export async function sendDonationThankYou(
  email: string,
  name: string,
  amount: string,
  currency: string,
  category?: string
) {
  const html = `
    <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
      <h2 style="color: #059669;">Thank You for Your Generous Donation</h2>
      <p>Dear ${name},</p>
      <p>Thank you for your generous donation to Setaragan Zirak School!</p>
      <div style="background-color: #f0fdf4; padding: 15px; border-radius: 5px; margin: 20px 0; border-left: 4px solid #10b981;">
        <p><strong>Donation Details:</strong></p>
        <p>Amount: <strong>${amount} ${currency}</strong></p>
        ${category ? `<p>Category: <strong>${category}</strong></p>` : ''}
        <p style="color: #6b7280; font-size: 12px; margin: 10px 0 0 0;">
          Your donation will help us provide quality education and support to our students.
        </p>
      </div>
      <p>A receipt for your records will be sent separately if applicable.</p>
      <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 20px 0;" />
      <p style="color: #6b7280; font-size: 12px;">
        <strong>Setaragan Zirak School</strong><br>
        Market Safid Street, Kabul, Afghanistan<br>
        Phone: +93747500432
      </p>
    </div>
  `;

  return sendEmail({
    to: email,
    subject: `Donation Receipt - ${amount} ${currency} - Setaragan Zirak School`,
    html,
  });
}

/**
 * Send admissions inquiry confirmation
 */
export async function sendAdmissionsConfirmation(
  parentEmail: string,
  parentName: string,
  studentName: string,
  gradeLevel: string
) {
  const html = `
    <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
      <h2 style="color: #1e40af;">Admissions Inquiry Received</h2>
      <p>Dear ${parentName},</p>
      <p>Thank you for your interest in Setaragan Zirak School! We have received your admissions inquiry for ${studentName}.</p>
      <div style="background: #eff6ff; padding: 15px; border-radius: 5px; margin: 15px 0; border-left: 3px solid #3b82f6;">
        <p><strong>Student Name:</strong> ${studentName}</p>
        <p><strong>Requested Grade:</strong> ${gradeLevel}</p>
      </div>
      <p>Our admissions team will review your application and contact you within 2-3 business days with next steps.</p>
      <p>If you have any questions in the meantime, please don't hesitate to reach out.</p>
      <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 20px 0;" />
      <p style="color: #6b7280; font-size: 12px;">
        <strong>Setaragan Zirak School</strong><br>
        Phone: +93747500432<br>
        Email: admissions@domain.com
      </p>
    </div>
  `;

  return sendEmail({
    to: parentEmail,
    subject: `Admissions Inquiry Received - ${studentName}`,
    html,
  });
}

/**
 * Send password reset email
 */
export async function sendPasswordResetEmail(
  email: string,
  resetUrl: string,
  userName: string = "User"
) {
  const html = `
    <div style="font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; color: #333;">
      <div style="background: linear-gradient(135deg, #1e40af 0%, #7c3aed 100%); padding: 30px; text-align: center; border-radius: 8px 8px 0 0;">
        <h1 style="color: white; margin: 0; font-size: 28px;">Password Reset Request</h1>
      </div>
      
      <div style="background: #f9fafb; padding: 30px; border: 1px solid #e5e7eb; border-radius: 0 0 8px 8px;">
        <p style="color: #374151; font-size: 16px; margin: 0 0 20px 0;">
          Hello <strong>${userName}</strong>,
        </p>
        
        <p style="color: #374151; font-size: 14px; line-height: 1.6; margin: 0 0 20px 0;">
          We received a request to reset the password for your Setaragan Zirak School account. 
          Click the button below to create a new password.
        </p>
        
        <div style="text-align: center; margin: 30px 0;">
          <a href="${resetUrl}" 
             style="background: linear-gradient(135deg, #1e40af 0%, #7c3aed 100%); 
                    color: white; 
                    padding: 12px 32px; 
                    text-decoration: none; 
                    border-radius: 6px; 
                    display: inline-block; 
                    font-weight: 600;
                    font-size: 16px;">
            Reset Password
          </a>
        </div>
        
        <div style="background: #eff6ff; padding: 15px; border-radius: 6px; margin: 20px 0; border-left: 3px solid #3b82f6;">
          <p style="font-size: 12px; color: #1e40af; margin: 0;">
            <strong>Or copy and paste this link in your browser:</strong><br>
            <span style="word-break: break-all; color: #0369a1;">${resetUrl}</span>
          </p>
        </div>
        
        <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 30px 0;">
        
        <div style="background: #fef3c7; padding: 15px; border-radius: 6px; border-left: 3px solid #f59e0b; margin: 20px 0;">
          <p style="font-size: 13px; color: #92400e; margin: 0;">
            <strong>⏰ Important:</strong> This link will expire in <strong>15 minutes</strong>. 
            If you didn't request a password reset, please ignore this email or contact our support team.
          </p>
        </div>
        
        <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 30px 0;">
        
        <p style="font-size: 12px; color: #6b7280; margin: 0 0 10px 0; line-height: 1.6;">
          <strong>Setaragan Zirak School</strong><br>
          Market Safid Street, Kabul, Afghanistan<br>
          Phone: +93 (0) 747 500 432<br>
          Email: support@setaraganzirak.edu.af
        </p>
        
        <p style="font-size: 11px; color: #9ca3af; margin: 15px 0 0 0;">
          This is an automated message. Please do not reply to this email.
        </p>
      </div>
    </div>
  `;

  return sendEmail({
    to: email,
    subject: 'Password Reset Request - Setaragan Zirak School',
    html,
  });
}

/**
 * Send user approval email
 */
export async function sendUserApprovalEmail(
  email: string,
  userName: string,
  loginUrl: string
) {
  const html = `
    <div style="font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; color: #333;">
      <div style="background: linear-gradient(135deg, #059669 0%, #10b981 100%); padding: 30px; text-align: center; border-radius: 8px 8px 0 0;">
        <h1 style="color: white; margin: 0; font-size: 28px;">✓ Account Approved!</h1>
      </div>
      
      <div style="background: #f9fafb; padding: 30px; border: 1px solid #e5e7eb; border-radius: 0 0 8px 8px;">
        <p style="color: #374151; font-size: 16px; margin: 0 0 20px 0;">
          Hello <strong>${userName}</strong>,
        </p>
        
        <p style="color: #374151; font-size: 14px; line-height: 1.6; margin: 0 0 20px 0;">
          Great news! Your admin account has been approved by our administrator. 
          You can now access the admin panel and begin managing content.
        </p>
        
        <div style="text-align: center; margin: 30px 0;">
          <a href="${loginUrl}" 
             style="background: linear-gradient(135deg, #059669 0%, #10b981 100%); 
                    color: white; 
                    padding: 12px 32px; 
                    text-decoration: none; 
                    border-radius: 6px; 
                    display: inline-block; 
                    font-weight: 600;
                    font-size: 16px;">
            Go to Admin Panel
          </a>
        </div>
        
        <div style="background: #ecfdf5; padding: 15px; border-radius: 6px; margin: 20px 0; border-left: 3px solid #10b981;">
          <p style="font-size: 12px; color: #065f46; margin: 0;">
            <strong>Login URL:</strong> <a href="${loginUrl}" style="color: #059669; text-decoration: none;">${loginUrl}</a>
          </p>
        </div>
        
        <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 30px 0;">
        
        <p style="font-size: 12px; color: #6b7280; margin: 0 0 10px 0; line-height: 1.6;">
          <strong>Setaragan Zirak School</strong><br>
          Admin Support Team<br>
          Phone: +93 (0) 747 500 432
        </p>
      </div>
    </div>
  `;

  return sendEmail({
    to: email,
    subject: 'Account Approved - Welcome to Setaragan Admin Panel',
    html,
  });
}

/**
 * Send user rejection email
 */
export async function sendUserRejectionEmail(
  email: string,
  userName: string,
  reason?: string
) {
  const html = `
    <div style="font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; color: #333;">
      <div style="background: linear-gradient(135deg, #dc2626 0%, #ef4444 100%); padding: 30px; text-align: center; border-radius: 8px 8px 0 0;">
        <h1 style="color: white; margin: 0; font-size: 28px;">Application Decision</h1>
      </div>
      
      <div style="background: #f9fafb; padding: 30px; border: 1px solid #e5e7eb; border-radius: 0 0 8px 8px;">
        <p style="color: #374151; font-size: 16px; margin: 0 0 20px 0;">
          Hello <strong>${userName}</strong>,
        </p>
        
        <p style="color: #374151; font-size: 14px; line-height: 1.6; margin: 0 0 20px 0;">
          Thank you for your interest in joining the Setaragan Zirak School admin team. 
          Unfortunately, your application has not been approved at this time.
        </p>
        
        ${reason ? `
        <div style="background: #fef2f2; padding: 15px; border-radius: 6px; margin: 20px 0; border-left: 3px solid #dc2626;">
          <p style="font-size: 13px; color: #7f1d1d; margin: 0;">
            <strong>Reason:</strong> ${reason}
          </p>
        </div>
        ` : ''}
        
        <p style="color: #374151; font-size: 14px; line-height: 1.6; margin: 20px 0 0 0;">
          If you have any questions or would like more information, please contact our support team.
        </p>
        
        <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 30px 0;">
        
        <p style="font-size: 12px; color: #6b7280; margin: 0 0 10px 0; line-height: 1.6;">
          <strong>Setaragan Zirak School</strong><br>
          Admin Support Team<br>
          Email: support@setaraganzirak.edu.af<br>
          Phone: +93 (0) 747 500 432
        </p>
      </div>
    </div>
  `;

  return sendEmail({
    to: email,
    subject: 'Application Decision - Setaragan Admin Panel',
    html,
  });
}

/**
 * Send user suspension email
 */
export async function sendUserSuspensionEmail(
  email: string,
  userName: string,
  reason?: string
) {
  const html = `
    <div style="font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; color: #333;">
      <div style="background: linear-gradient(135deg, #ea580c 0%, #f97316 100%); padding: 30px; text-align: center; border-radius: 8px 8px 0 0;">
        <h1 style="color: white; margin: 0; font-size: 28px;">Account Suspended</h1>
      </div>
      
      <div style="background: #f9fafb; padding: 30px; border: 1px solid #e5e7eb; border-radius: 0 0 8px 8px;">
        <p style="color: #374151; font-size: 16px; margin: 0 0 20px 0;">
          Hello <strong>${userName}</strong>,
        </p>
        
        <p style="color: #374151; font-size: 14px; line-height: 1.6; margin: 0 0 20px 0;">
          Your admin account has been temporarily suspended.
        </p>
        
        ${reason ? `
        <div style="background: #fff7ed; padding: 15px; border-radius: 6px; margin: 20px 0; border-left: 3px solid #f97316;">
          <p style="font-size: 13px; color: #7c2d12; margin: 0;">
            <strong>Reason:</strong> ${reason}
          </p>
        </div>
        ` : ''}
        
        <p style="color: #374151; font-size: 14px; line-height: 1.6; margin: 20px 0 0 0;">
          Please contact the administrator to discuss this action or request reactivation.
        </p>
        
        <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 30px 0;">
        
        <p style="font-size: 12px; color: #6b7280; margin: 0 0 10px 0; line-height: 1.6;">
          <strong>Setaragan Zirak School</strong><br>
          Admin Support Team<br>
          Email: support@setaraganzirak.edu.af<br>
          Phone: +93 (0) 747 500 432
        </p>
      </div>
    </div>
  `;

  return sendEmail({
    to: email,
    subject: 'Account Suspended - Setaragan Admin Panel',
    html,
  });
}

/**
 * Send account reactivation email
 */
export async function sendAccountReactivationEmail(
  email: string,
  userName: string,
  loginUrl: string
) {
  const html = `
    <div style="font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; color: #333;">
      <div style="background: linear-gradient(135deg, #059669 0%, #10b981 100%); padding: 30px; text-align: center; border-radius: 8px 8px 0 0;">
        <h1 style="color: white; margin: 0; font-size: 28px;">✓ Account Reactivated!</h1>
      </div>
      
      <div style="background: #f9fafb; padding: 30px; border: 1px solid #e5e7eb; border-radius: 0 0 8px 8px;">
        <p style="color: #374151; font-size: 16px; margin: 0 0 20px 0;">
          Hello <strong>${userName}</strong>,
        </p>
        
        <p style="color: #374151; font-size: 14px; line-height: 1.6; margin: 0 0 20px 0;">
          Your admin account has been reactivated and you can now access the admin panel again.
        </p>
        
        <div style="text-align: center; margin: 30px 0;">
          <a href="${loginUrl}" 
             style="background: linear-gradient(135deg, #059669 0%, #10b981 100%); 
                    color: white; 
                    padding: 12px 32px; 
                    text-decoration: none; 
                    border-radius: 6px; 
                    display: inline-block; 
                    font-weight: 600;
                    font-size: 16px;">
            Go to Admin Panel
          </a>
        </div>
        
        <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 30px 0;">
        
        <p style="font-size: 12px; color: #6b7280; margin: 0 0 10px 0; line-height: 1.6;">
          <strong>Setaragan Zirak School</strong><br>
          Admin Support Team<br>
          Phone: +93 (0) 747 500 432
        </p>
      </div>
    </div>
  `;

  return sendEmail({
    to: email,
    subject: 'Account Reactivated - Setaragan Admin Panel',
    html,
  });
}

/**
 * Test SMTP connection
 */
export async function testSMTPConnection(): Promise<{ success: boolean; message: string }> {
  try {
    if (!validateSMTPConfig()) {
      return {
        success: false,
        message: 'SMTP configuration is incomplete. Check your .env.local file.',
      };
    }

    const transporter = getTransporter();
    await transporter.verify();

    console.log('✅ SMTP connection verified successfully');
    return {
      success: true,
      message: 'SMTP connection verified successfully',
    };
  } catch (error) {
    const message = error instanceof Error ? error.message : 'Unknown error';
    console.error('❌ SMTP connection failed:', message);
    return {
      success: false,
      message: `SMTP connection failed: ${message}`,
    };
  }
}

