import * as Yup from "yup";

// Contact Form Schema
export const CONTACT_SCHEMA = Yup.object().shape({
  name: Yup.string()
    .required("Name is required")
    .min(2, "Name must be at least 2 characters")
    .max(100, "Name must not exceed 100 characters"),
  email: Yup.string()
    .email("Please enter a valid email address")
    .required("Email is required"),
  phone: Yup.string()
    .optional()
    .matches(/^[+]?[(]?[0-9]{3}[)]?[-\s.]?[0-9]{3}[-\s.]?[0-9]{4,6}$/, "Please enter a valid phone number"),
  subject: Yup.string()
    .required("Please select a subject")
    .oneOf(["general", "admission", "technical", "other"], "Invalid subject"),
  message: Yup.string()
    .required("Message is required")
    .min(20, "Message must be at least 20 characters")
    .max(5000, "Message must not exceed 5000 characters"),
});

// Donation Form Schema
export const DONATION_SCHEMA = Yup.object().shape({
  donorName: Yup.string()
    .required("Full name is required")
    .min(2, "Name must be at least 2 characters")
    .max(100, "Name must not exceed 100 characters"),
  email: Yup.string()
    .email("Please enter a valid email address")
    .required("Email is required"),
  phone: Yup.string()
    .optional()
    .matches(/^[+]?[(]?[0-9]{3}[)]?[-\s.]?[0-9]{3}[-\s.]?[0-9]{4,6}$/, "Please enter a valid phone number"),
  amount: Yup.number()
    .required("Amount is required")
    .min(10, "Minimum donation amount is $10")
    .max(100000, "Maximum donation amount is $100,000"),
  currency: Yup.string()
    .required("Currency is required")
    .oneOf(["USD", "EUR", "GBP", "AFN", "CAD"], "Invalid currency"),
  category: Yup.string()
    .required("Please select a donation category")
    .oneOf(["education", "infrastructure", "scholarships", "sports", "arts", "general"], "Invalid category"),
  message: Yup.string()
    .optional()
    .max(1000, "Message must not exceed 1000 characters"),
  paymentMethod: Yup.string()
    .required("Please select a payment method")
    .oneOf(["card", "transfer", "paypal", "crypto"], "Invalid payment method"),
  anonymous: Yup.boolean().optional(),
});

// Newsletter Subscription Schema
export const NEWSLETTER_SCHEMA = Yup.object().shape({
  email: Yup.string()
    .email("Please enter a valid email address")
    .required("Email is required"),
  firstName: Yup.string()
    .optional()
    .max(50, "First name must not exceed 50 characters"),
});

// Admissions Inquiry Schema
export const ADMISSIONS_SCHEMA = Yup.object().shape({
  studentName: Yup.string()
    .required("Student name is required")
    .min(2, "Name must be at least 2 characters")
    .max(100, "Name must not exceed 100 characters"),
  gradeLevel: Yup.string()
    .required("Please select a grade level")
    .oneOf(["kg", "g1", "g2", "g3", "g4", "g5", "g6", "g7", "g8", "g9", "g10", "g11", "g12"], "Invalid grade level"),
  parentName: Yup.string()
    .required("Parent/Guardian name is required")
    .min(2, "Name must be at least 2 characters")
    .max(100, "Name must not exceed 100 characters"),
  parentEmail: Yup.string()
    .email("Please enter a valid email address")
    .required("Parent email is required"),
  parentPhone: Yup.string()
    .required("Parent phone is required")
    .matches(/^[+]?[(]?[0-9]{3}[)]?[-\s.]?[0-9]{3}[-\s.]?[0-9]{4,6}$/, "Please enter a valid phone number"),
  message: Yup.string()
    .optional()
    .max(1000, "Message must not exceed 1000 characters"),
  hearAbout: Yup.string()
    .optional()
    .oneOf(["social-media", "friend", "google", "school-fair", "other"], "Invalid option"),
});

// General Inquiry Schema
export const INQUIRY_SCHEMA = Yup.object().shape({
  name: Yup.string()
    .required("Name is required")
    .min(2, "Name must be at least 2 characters")
    .max(100, "Name must not exceed 100 characters"),
  email: Yup.string()
    .email("Please enter a valid email address")
    .required("Email is required"),
  topic: Yup.string()
    .required("Please select a topic"),
  message: Yup.string()
    .required("Message is required")
    .min(20, "Message must be at least 20 characters")
    .max(2000, "Message must not exceed 2000 characters"),
});

// Export all schemas
export const FORM_SCHEMAS = {
  contact: CONTACT_SCHEMA,
  donation: DONATION_SCHEMA,
  newsletter: NEWSLETTER_SCHEMA,
  admissions: ADMISSIONS_SCHEMA,
  inquiry: INQUIRY_SCHEMA,
};

// Constants for form options
export const CONTACT_SUBJECTS = [
  { value: "general", label: "General Inquiry" },
  { value: "admission", label: "Admission Question" },
  { value: "donation", label: "Donation Inquiry" },
  { value: "technical", label: "Technical Issue" },
  { value: "other", label: "Other" },
];

export const DONATION_CATEGORIES = [
  { value: "education", label: "Education Programs" },
  { value: "infrastructure", label: "School Infrastructure" },
  { value: "scholarships", label: "Student Scholarships" },
  { value: "sports", label: "Sports Programs" },
  { value: "arts", label: "Arts & Culture" },
  { value: "general", label: "General Fund" },
];

export const PAYMENT_METHODS = [
  { value: "card", label: "Credit/Debit Card" },
  { value: "transfer", label: "Bank Transfer" },
  { value: "paypal", label: "PayPal" },
  { value: "crypto", label: "Cryptocurrency" },
];

export const CURRENCIES = [
  { value: "USD", label: "US Dollar ($)" },
  { value: "EUR", label: "Euro (€)" },
  { value: "GBP", label: "British Pound (£)" },
  { value: "AFN", label: "Afghan Afghani (؋)" },
  { value: "CAD", label: "Canadian Dollar (C$)" },
];

export const GRADE_LEVELS = [
  { value: "kg", label: "Kindergarten" },
  { value: "g1", label: "Grade 1" },
  { value: "g2", label: "Grade 2" },
  { value: "g3", label: "Grade 3" },
  { value: "g4", label: "Grade 4" },
  { value: "g5", label: "Grade 5" },
  { value: "g6", label: "Grade 6" },
  { value: "g7", label: "Grade 7" },
  { value: "g8", label: "Grade 8" },
  { value: "g9", label: "Grade 9" },
  { value: "g10", label: "Grade 10" },
  { value: "g11", label: "Grade 11" },
  { value: "g12", label: "Grade 12" },
];

export const HEAR_ABOUT_US = [
  { value: "social-media", label: "Social Media" },
  { value: "friend", label: "Friend/Family" },
  { value: "google", label: "Google Search" },
  { value: "school-fair", label: "School Fair/Event" },
  { value: "other", label: "Other" },
];

// ==================== LEADERSHIP SCHEMA ====================
export const leadershipSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(3).max(100),
  position: Yup.string().required('Position is required').min(2).max(100),
  shortDescription: Yup.string().max(300, 'Short description should be max 300 characters'),
  bio: Yup.string().max(2000),
  image: Yup.string().nullable().test('is-valid-image-url', 'Invalid image URL', (value) => {
    if (!value) return true;
    return value.startsWith('/') || value.startsWith('http://') || value.startsWith('https://');
  }),
  expertise: Yup.string().max(2000),
  order: Yup.number().min(0).max(1000),
  active: Yup.boolean(),
});

// ==================== TEACHER SCHEMA ====================
export const teacherSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(3).max(100),
  title: Yup.string().required('Title is required').min(2).max(50),
  position: Yup.string().required('Position/Subject is required').min(2).max(100),
  subject: Yup.string().max(100).nullable(),
  gender: Yup.string().oneOf(['male', 'female', ''], 'Invalid gender').nullable(),
  bio: Yup.string().max(2000).nullable(),
  qualifications: Yup.string().max(2000).nullable(),
  experience: Yup.number().min(0).max(70).nullable(),
  image: Yup.string().nullable().test('is-valid-image-url', 'Invalid image URL', (value) => {
    if (!value) return true;
    return value.startsWith('/') || value.startsWith('http://') || value.startsWith('https://');
  }),
  email: Yup.string().email('Invalid email').nullable(),
  phone: Yup.string().max(20).nullable(),
  active: Yup.boolean(),
});

// ==================== MANUAL DONATION SCHEMA ====================
export const manualDonationSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(2),
  email: Yup.string().email().required('Email is required'),
  phone: Yup.string().optional(),
  country: Yup.string().optional(),
  amount: Yup.number().required('Amount is required').positive(),
  currency: Yup.string().required('Currency is required'),
  category: Yup.string().optional(),
  paymentMethod: Yup.string().required('Payment method is required'),
  paymentStatus: Yup.string()
    .required('Payment status is required')
    .oneOf(['pending', 'completed', 'failed']),
  anonymous: Yup.boolean().optional(),
  message: Yup.string().optional(),
  transactionId: Yup.string().optional(),
});
