/**
 * Unified Typography System using Tailwind CSS classes
 * 
 * This file contains standardized Tailwind typography classes for consistent
 * text styling across the entire application. All text elements follow a
 * unified scale and responsive design principles.
 */

// Typography Scale Configuration
export const typography = {
  // Display headings (Hero sections, major landing elements)
  display: {
    '1': 'text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-black tracking-tight leading-none',
    '2': 'text-2xl sm:text-3xl md:text-4xl font-bold tracking-tight leading-tight',
    '3': 'text-2xl sm:text-3xl md:text-4xl lg:text-5xl font-bold tracking-tight leading-tight'
  },

  // Standard headings (Section titles, content headings)
  heading: {
    '1': 'text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight',
    '2': 'text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight',
    '3': 'text-xl sm:text-2xl md:text-3xl font-bold leading-snug tracking-tight',
    '4': 'text-lg sm:text-xl md:text-2xl font-bold leading-snug',
    '5': 'text-base sm:text-lg md:text-xl font-bold leading-snug',
    '6': 'text-sm sm:text-base md:text-lg font-bold leading-snug'
  },

  // Body text (Paragraphs, descriptions)
  body: {
    'xl': 'text-lg sm:text-xl md:text-2xl leading-relaxed',
    'lg': 'text-base sm:text-lg md:text-xl leading-relaxed',
    'md': 'text-sm sm:text-base md:text-lg leading-relaxed',
    'sm': 'text-xs sm:text-sm md:text-base leading-relaxed',
    'xs': 'text-xs sm:text-sm leading-relaxed'
  },

  // Labels and captions
  label: {
    'lg': 'text-sm sm:text-base font-semibold tracking-wide',
    'md': 'text-xs sm:text-sm font-semibold tracking-wide',
    'sm': 'text-xs font-semibold tracking-wide uppercase'
  },

  // Buttons and interactive elements
  button: {
    'lg': 'text-base sm:text-lg font-semibold',
    'md': 'text-sm sm:text-base font-semibold',
    'sm': 'text-xs sm:text-sm font-semibold'
  },

  // Statistics and numbers
  stat: {
    'xl': 'text-2xl sm:text-3xl md:text-4xl font-black',
    'lg': 'text-2xl sm:text-3xl md:text-4xl font-black',
    'md': 'text-xl sm:text-2xl md:text-3xl font-black',
    'sm': 'text-lg sm:text-xl md:text-2xl font-black'
  },

  // Testimonials and quotes
  quote: {
    'lg': 'text-xl sm:text-2xl md:text-3xl font-light italic leading-relaxed',
    'md': 'text-lg sm:text-xl md:text-2xl font-light italic leading-relaxed',
    'sm': 'text-base sm:text-lg md:text-xl font-light italic leading-relaxed'
  },

  // Navigation and menu items
  nav: {
    'primary': 'text-base font-medium',
    'secondary': 'text-sm font-medium',
    'mobile': 'text-lg font-medium'
  },

  // Forms and inputs
  form: {
    'label': 'text-sm font-semibold text-gray-700',
    'input': 'text-base',
    'help': 'text-sm text-gray-500',
    'error': 'text-sm text-red-600 font-medium'
  },

  // Cards and components
  card: {
    'title': 'text-xl sm:text-2xl font-bold leading-snug',
    'subtitle': 'text-base sm:text-lg font-medium text-gray-600',
    'body': 'text-sm sm:text-base leading-relaxed text-gray-600'
  },

  // Badges and tags
  badge: {
    'lg': 'text-sm font-semibold px-3 py-1',
    'md': 'text-xs font-semibold px-2.5 py-1',
    'sm': 'text-xs font-medium px-2 py-0.5'
  },

  // Lists
  list: {
    'item': 'text-base leading-relaxed',
    'description': 'text-sm text-gray-600'
  }
} as const;

// Color combinations for text
export const textColors = {
  primary: 'text-gray-900',
  secondary: 'text-gray-600', 
  muted: 'text-gray-500',
  inverse: 'text-white',
  brand: 'text-primary-600',
  accent: 'text-secondary-600',
  success: 'text-green-600',
  warning: 'text-orange-600',
  error: 'text-red-600',
  info: 'text-blue-600'
} as const;

// Gradients for special text effects
export const textGradients = {
  primary: 'bg-gradient-to-r from-primary-600 to-primary-500 bg-clip-text text-transparent',
  secondary: 'bg-gradient-to-r from-secondary-600 to-secondary-500 bg-clip-text text-transparent',
  brand: 'bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 bg-clip-text text-transparent',
  warm: 'bg-gradient-to-r from-orange-600 to-red-600 bg-clip-text text-transparent',
  cool: 'bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent'
} as const;

// Utility function to get typography classes
export function getTypographyClass(
  category: keyof typeof typography,
  size: string,
  color?: keyof typeof textColors | keyof typeof textGradients,
  gradient = false
): string {
  const baseClass = typography[category]?.[size as keyof typeof typography[typeof category]] || '';
  
  if (!color) return baseClass;
  
  if (gradient && color in textGradients) {
    return `${baseClass} ${textGradients[color as keyof typeof textGradients]}`;
  }
  
  if (color in textColors) {
    return `${baseClass} ${textColors[color as keyof typeof textColors]}`;
  }
  
  return baseClass;
}

// Responsive utility classes
export const responsive = {
  container: 'max-w-8xl mx-auto px-4 sm:px-6 lg:px-8',
  section: 'py-16 sm:py-20 lg:py-28',
  grid: {
    '1': 'grid grid-cols-1',
    '2': 'grid grid-cols-1 md:grid-cols-2',
    '3': 'grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3',
    '4': 'grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4'
  },
  gaps: {
    'sm': 'gap-4 sm:gap-6',
    'md': 'gap-6 sm:gap-8',
    'lg': 'gap-8 sm:gap-12',
    'xl': 'gap-12 sm:gap-16'
  }
} as const;

// Common component patterns
export const patterns = {
  badge: 'inline-flex items-center gap-2 rounded-full font-semibold',
  card: 'bg-white rounded-lg shadow-lg border border-gray-200/50',
  button: 'inline-flex items-center justify-center rounded-lg transition-all duration-200 focus:outline-none focus:ring-4',
  heroSection: 'relative py-20 lg:py-28 overflow-hidden',
  section: 'py-16 sm:py-20 lg:py-28'
} as const;

export default typography;