import { auth } from "@/lib/auth";
import { NextResponse } from "next/server";
import type { NextRequest } from "next/server";
import { ROLES, UserRole } from "./lib/admin/permissions";

export async function middleware(request: NextRequest) {
  const session = await auth();
  const { pathname } = request.nextUrl;

  // ==================== ADMIN PAGE ROUTES ====================
  // Protect all admin UI routes
  if (pathname.startsWith("/admin")) {
    if (!session?.user) {
      return NextResponse.redirect(new URL("/auth/signin", request.url));
    }

    // Verify admin role
    if (!ROLES.includes(session.user.role as UserRole)) {
      return NextResponse.redirect(new URL("/auth/error?error=unauthorized", request.url));
    }
  }

  // ==================== ADMIN API ROUTES ====================
  // Protect all /api/admin/* routes
  if (pathname.startsWith("/api/admin")) {
    if (!session?.user?.id) {
      return NextResponse.json(
        { success: false, message: "Unauthorized" },
        { status: 401 }
      );
    }

    // Verify admin role
    if (!ROLES.includes(session.user.role as UserRole)) {
      return NextResponse.json(
        { success: false, message: "Forbidden - Admin access required" },
        { status: 403 }
      );
    }

    // Add user info to request headers for API routes
    const requestHeaders = new Headers(request.headers);
    requestHeaders.set("x-user-id", session.user.id);
    requestHeaders.set("x-user-role", session.user.role);

    return NextResponse.next({
      request: {
        headers: requestHeaders,
      },
    });
  }

  // ==================== AUTH ROUTES ====================
  // Redirect authenticated users away from auth pages
  if (pathname.startsWith("/auth") && session?.user) {
    return NextResponse.redirect(new URL("/admin", request.url));
  }

  return NextResponse.next();
}

export const config = {
  matcher: ["/admin/:path*", "/auth/:path*", "/api/admin/:path*"],
};
